import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Textarea } from '@/components/ui/textarea';

interface Activity {
  timestamp: number;
  url: string;
  title: string;
  action: string;
  element?: string;
  text?: string;
  tabId: number;
}

interface ProcessResult {
  success: boolean;
  sessionId: string;
  relevanceScore: number;
  shouldKeep: boolean;
  updatedPlan: {
    summary: string;
    steps: Array<{
      number: number;
      title: string;
      description: string;
      url?: string;
      duration?: string;
      completed: boolean;
    }>;
    confidence: number;
    lastUpdated: number;
  };
}

export default function IncrementalTestPage() {
  const [sessionId, setSessionId] = useState('test_session_' + Date.now());
  const [taskDescription, setTaskDescription] = useState('Creating a project presentation and sharing it with my team');
  const [currentPlan, setCurrentPlan] = useState<ProcessResult['updatedPlan'] | null>(null);
  const [activityLog, setActivityLog] = useState<Array<{ activity: Activity; result: ProcessResult }>>([]);
  const [isProcessing, setIsProcessing] = useState(false);
  const [recordingId, setRecordingId] = useState<string | null>(null);

  // Realistic activities: user creates a presentation and looks for images
  const predefinedActivities = [
    {
      timestamp: Date.now(),
      url: 'https://docs.google.com/presentation/d/abc123/edit',
      title: 'Project Kickoff - Google Slides',
      action: 'create',
      tabId: 1
    },
    {
      timestamp: Date.now() + 1000,
      url: 'https://docs.google.com/presentation/d/abc123/edit',
      title: 'Project Kickoff - Google Slides',
      action: 'edit',
      tabId: 1
    },
    {
      timestamp: Date.now() + 2000,
      url: 'https://unsplash.com/photos/business-meeting',
      title: 'Unsplash - Free Business Photos',
      action: 'navigate',
      tabId: 2
    },
    {
      timestamp: Date.now() + 3000,
      url: 'https://unsplash.com/download/photo123',
      title: 'Download Business Meeting Photo',
      action: 'download',
      tabId: 2
    }
  ];

  const processActivity = async (activity: Activity) => {
    setIsProcessing(true);
    try {
      const response = await fetch('/api/process-activity', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          sessionId,
          activity,
          taskDescription: taskDescription || undefined,
        }),
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const result: ProcessResult = await response.json();
      
      setActivityLog(prev => [...prev, { activity, result }]);
      if (result.shouldKeep) {
        setCurrentPlan(result.updatedPlan);
      }

      return result;
    } catch (error) {
      console.error('Failed to process activity:', error);
      throw error;
    } finally {
      setIsProcessing(false);
    }
  };

  const runAllActivities = async () => {
    setActivityLog([]);
    setCurrentPlan(null);
    
    for (const activity of predefinedActivities) {
      await processActivity(activity);
      // Small delay to see incremental updates
      await new Promise(resolve => setTimeout(resolve, 500));
    }
  };

  const createHolisticRecording = async () => {
    setIsProcessing(true);
    try {
      const response = await fetch('/api/recordings', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          taskDescription,
          capturedData: {
            sessionId,
            startTime: Date.now() - 3600000, // 1 hour ago
            endTime: Date.now(),
            activities: predefinedActivities,
            tabs: [
              {id: 1, url: 'https://docs.google.com/presentation/d/abc123/edit', title: 'Project Kickoff - Google Slides'},
              {id: 2, url: 'https://unsplash.com/photos/business-meeting', title: 'Unsplash - Free Business Photos'}
            ]
          }
        }),
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const result = await response.json();
      setCurrentPlan(result.plan);
      setRecordingId(result.recording.id);
    } catch (error) {
      console.error('Failed to create holistic recording:', error);
    } finally {
      setIsProcessing(false);
    }
  };

  const resetSession = () => {
    setSessionId('test_session_' + Date.now());
    setActivityLog([]);
    setCurrentPlan(null);
    setTaskDescription('');
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="text-center space-y-2">
        <h1 className="text-3xl font-bold">Incremental Planning Test</h1>
        <p className="text-muted-foreground">
          Test the AI-powered incremental planning system with real-time activity filtering
        </p>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Controls */}
        <Card>
          <CardHeader>
            <CardTitle>Test Controls</CardTitle>
            <CardDescription>Configure and run incremental planning tests</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <label className="text-sm font-medium">Session ID</label>
              <Input value={sessionId} onChange={(e) => setSessionId(e.target.value)} />
            </div>

            <div>
              <label className="text-sm font-medium">Task Description</label>
              <Textarea 
                placeholder="Describe what the user is trying to accomplish..."
                value={taskDescription}
                onChange={(e) => setTaskDescription(e.target.value)}
              />
            </div>

            <div className="flex gap-2 flex-wrap">
              <Button onClick={createHolisticRecording} disabled={isProcessing}>
                Test Holistic Planning
              </Button>
              <Button onClick={resetSession} variant="outline">
                Reset
              </Button>
              {recordingId && (
                <Button 
                  onClick={() => window.open(`/feedback/${recordingId}`, '_blank')} 
                  variant="secondary"
                >
                  Provide Feedback
                </Button>
              )}
            </div>

            {isProcessing && (
              <div className="text-sm text-muted-foreground">
                Processing activity...
              </div>
            )}
          </CardContent>
        </Card>

        {/* Current Plan */}
        <Card>
          <CardHeader>
            <CardTitle>Current Plan</CardTitle>
            <CardDescription>Real-time plan updates from incremental planner</CardDescription>
          </CardHeader>
          <CardContent>
            {currentPlan ? (
              <div className="space-y-4">
                <div>
                  <h3 className="font-semibold">{currentPlan.summary}</h3>
                  <div className="flex gap-2 mt-2">
                    <Badge variant="secondary">
                      Confidence: {Math.round(currentPlan.confidence * 100)}%
                    </Badge>
                    <Badge variant="outline">
                      {currentPlan.steps.length} steps
                    </Badge>
                  </div>
                </div>

                <div className="space-y-2">
                  {currentPlan.steps.map((step) => (
                    <div key={step.number} className="border rounded-lg p-3">
                      <div className="flex items-start gap-3">
                        <div className="w-6 h-6 rounded-full bg-primary text-primary-foreground text-sm flex items-center justify-center">
                          {step.number}
                        </div>
                        <div className="flex-1">
                          <h4 className="font-medium">{step.title}</h4>
                          <p className="text-sm text-muted-foreground">{step.description}</p>
                          {step.duration && (
                            <Badge variant="outline" className="mt-1">
                              {step.duration}
                            </Badge>
                          )}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            ) : (
              <div className="text-center text-muted-foreground py-8">
                No plan generated yet. Run test activities to see incremental planning in action.
              </div>
            )}
          </CardContent>
        </Card>
      </div>

      {/* Activity Log */}
      <Card>
        <CardHeader>
          <CardTitle>Activity Processing Log</CardTitle>
          <CardDescription>See how each activity is processed and filtered</CardDescription>
        </CardHeader>
        <CardContent>
          {activityLog.length > 0 ? (
            <div className="space-y-3">
              {activityLog.map((entry, index) => (
                <div key={index} className="border rounded-lg p-4">
                  <div className="flex items-center justify-between mb-2">
                    <div className="font-medium">
                      {entry.activity.action} on {entry.activity.title}
                    </div>
                    <div className="flex gap-2">
                      <Badge variant={entry.result.shouldKeep ? "default" : "destructive"}>
                        Score: {entry.result.relevanceScore}
                      </Badge>
                      <Badge variant={entry.result.shouldKeep ? "default" : "secondary"}>
                        {entry.result.shouldKeep ? "Kept" : "Filtered"}
                      </Badge>
                    </div>
                  </div>
                  <div className="text-sm text-muted-foreground">
                    URL: {entry.activity.url}
                  </div>
                  {entry.activity.element && (
                    <div className="text-sm text-muted-foreground">
                      Element: {entry.activity.element}
                    </div>
                  )}
                  {entry.activity.text && (
                    <div className="text-sm text-muted-foreground">
                      Text: "{entry.activity.text}"
                    </div>
                  )}
                </div>
              ))}
            </div>
          ) : (
            <div className="text-center text-muted-foreground py-8">
              No activities processed yet. Click "Run Test Activities" to start.
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}