import { useState, useEffect } from 'react';
import { useLocation, useParams } from 'wouter';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Textarea } from '@/components/ui/textarea';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { ArrowLeft, ThumbsUp, ThumbsDown, MessageSquare, CheckCircle } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';

interface ActivityIntent {
  url: string;
  title: string;
  inferredIntent: string;
  confidence: number;
  reasoning: string;
  relevance: number;
  timestamp: number;
}

interface FeedbackItem {
  activityUrl: string;
  wasCorrect: boolean;
  userCorrection: string;
}

export default function FeedbackPage() {
  const params = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  
  const recordingId = params.id;
  const [activityIntents, setActivityIntents] = useState<ActivityIntent[]>([]);
  const [loading, setLoading] = useState(true);
  const [feedback, setFeedback] = useState<Map<string, FeedbackItem>>(new Map());
  const [submitting, setSubmitting] = useState(false);
  const [submitted, setSubmitted] = useState(false);

  useEffect(() => {
    if (recordingId) {
      fetchActivityIntents();
    }
  }, [recordingId]);

  const fetchActivityIntents = async () => {
    try {
      const response = await fetch(`/api/recordings/${recordingId}/intents`);
      if (!response.ok) {
        throw new Error('Failed to fetch activity intents');
      }
      const data = await response.json();
      setActivityIntents(data.activityIntents || []);
    } catch (error) {
      toast({
        variant: "destructive",
        title: "Error",
        description: "Failed to load activity intents",
      });
    } finally {
      setLoading(false);
    }
  };

  const updateFeedback = (url: string, updates: Partial<FeedbackItem>) => {
    setFeedback(prev => {
      const newFeedback = new Map(prev);
      const existing = newFeedback.get(url) || { activityUrl: url, wasCorrect: true, userCorrection: '' };
      newFeedback.set(url, { ...existing, ...updates });
      return newFeedback;
    });
  };

  const submitFeedback = async () => {
    setSubmitting(true);
    try {
      const feedbackData = Array.from(feedback.values()).filter(item => 
        item.wasCorrect === false || item.userCorrection.trim()
      );

      for (const item of feedbackData) {
        const intent = activityIntents.find(ai => ai.url === item.activityUrl);
        if (intent) {
          await fetch('/api/feedback', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
              sessionId: recordingId, // Using recordingId as sessionId for this demo
              activityUrl: item.activityUrl,
              inferredIntent: intent.inferredIntent,
              userCorrection: item.userCorrection,
              wasCorrect: item.wasCorrect,
            }),
          });
        }
      }

      setSubmitted(true);
      toast({
        title: "Feedback Submitted",
        description: "Thank you! Your feedback will help improve future activity recognition.",
      });
    } catch (error) {
      toast({
        variant: "destructive",
        title: "Submission Failed",
        description: "Failed to submit feedback. Please try again.",
      });
    } finally {
      setSubmitting(false);
    }
  };

  if (loading) {
    return (
      <div className="container mx-auto p-6">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary mx-auto"></div>
          <p className="mt-2 text-muted-foreground">Loading activity analysis...</p>
        </div>
      </div>
    );
  }

  if (submitted) {
    return (
      <div className="container mx-auto p-6">
        <Card className="max-w-2xl mx-auto text-center">
          <CardContent className="pt-6">
            <CheckCircle className="w-16 h-16 text-green-500 mx-auto mb-4" />
            <h2 className="text-2xl font-bold mb-2">Feedback Submitted!</h2>
            <p className="text-muted-foreground mb-6">
              Thank you for helping us improve the activity recognition system. 
              Your feedback will be used to enhance future plan generation.
            </p>
            <Button onClick={() => navigate('/')}>
              Return to Home
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 max-w-4xl">
      <div className="flex items-center gap-4 mb-6">
        <Button variant="ghost" size="icon" onClick={() => navigate('/')}>
          <ArrowLeft className="h-4 w-4" />
        </Button>
        <div>
          <h1 className="text-3xl font-bold">Activity Intent Feedback</h1>
          <p className="text-muted-foreground">
            Help us improve by reviewing what the AI understood from each activity
          </p>
        </div>
      </div>

      {activityIntents.length === 0 ? (
        <Card>
          <CardContent className="pt-6 text-center">
            <p className="text-muted-foreground">
              No activity intents found for this recording.
            </p>
          </CardContent>
        </Card>
      ) : (
        <div className="space-y-6">
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
            <h3 className="font-semibold text-blue-900 mb-2">How to provide feedback:</h3>
            <ul className="text-sm text-blue-800 space-y-1">
              <li>• Review what the AI thought you were doing on each website</li>
              <li>• Mark if the interpretation was correct or incorrect</li>
              <li>• If incorrect, provide the correct description of what you were actually doing</li>
              <li>• Your feedback helps improve future activity recognition</li>
            </ul>
          </div>

          {activityIntents.map((intent, index) => {
            const currentFeedback = feedback.get(intent.url);
            const domain = new URL(intent.url).hostname;

            return (
              <Card key={index} className="border-l-4 border-l-blue-500">
                <CardHeader>
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <CardTitle className="text-lg">{intent.title}</CardTitle>
                      <CardDescription className="mt-1">
                        {domain} • {new Date(intent.timestamp).toLocaleTimeString()}
                      </CardDescription>
                    </div>
                    <div className="flex gap-2">
                      <Badge variant="secondary">
                        Confidence: {Math.round(intent.confidence * 100)}%
                      </Badge>
                      <Badge variant={intent.relevance >= 0.7 ? "default" : "outline"}>
                        Relevance: {Math.round(intent.relevance * 100)}%
                      </Badge>
                    </div>
                  </div>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <h4 className="font-semibold text-sm text-gray-700 mb-2">
                      AI's interpretation of what you were doing:
                    </h4>
                    <div className="bg-gray-50 rounded-lg p-3">
                      <p className="font-medium">{intent.inferredIntent}</p>
                      <p className="text-sm text-muted-foreground mt-1">
                        {intent.reasoning}
                      </p>
                    </div>
                  </div>

                  <div className="space-y-3">
                    <Label className="text-sm font-semibold">
                      Was this interpretation correct?
                    </Label>
                    <RadioGroup
                      value={currentFeedback?.wasCorrect?.toString() || "true"}
                      onValueChange={(value) => 
                        updateFeedback(intent.url, { wasCorrect: value === "true" })
                      }
                    >
                      <div className="flex items-center space-x-2">
                        <RadioGroupItem value="true" id={`correct-${index}`} />
                        <Label htmlFor={`correct-${index}`} className="flex items-center gap-2">
                          <ThumbsUp className="w-4 h-4 text-green-600" />
                          Yes, this is accurate
                        </Label>
                      </div>
                      <div className="flex items-center space-x-2">
                        <RadioGroupItem value="false" id={`incorrect-${index}`} />
                        <Label htmlFor={`incorrect-${index}`} className="flex items-center gap-2">
                          <ThumbsDown className="w-4 h-4 text-red-600" />
                          No, this is incorrect
                        </Label>
                      </div>
                    </RadioGroup>

                    {currentFeedback?.wasCorrect === false && (
                      <div className="mt-3">
                        <Label htmlFor={`correction-${index}`} className="text-sm font-semibold">
                          What were you actually doing? <span className="text-red-500">*</span>
                        </Label>
                        <Textarea
                          id={`correction-${index}`}
                          placeholder="Describe what you were actually trying to accomplish on this website..."
                          value={currentFeedback?.userCorrection || ''}
                          onChange={(e) => 
                            updateFeedback(intent.url, { userCorrection: e.target.value })
                          }
                          className="mt-2"
                        />
                      </div>
                    )}

                    {currentFeedback?.wasCorrect === true && (
                      <div className="mt-3">
                        <Label htmlFor={`additional-${index}`} className="text-sm font-semibold">
                          Additional context (optional)
                        </Label>
                        <Textarea
                          id={`additional-${index}`}
                          placeholder="Any additional details that might help improve understanding..."
                          value={currentFeedback?.userCorrection || ''}
                          onChange={(e) => 
                            updateFeedback(intent.url, { userCorrection: e.target.value })
                          }
                          className="mt-2"
                        />
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            );
          })}

          <div className="flex justify-between items-center pt-6 border-t">
            <Button variant="outline" onClick={() => navigate('/')}>
              Skip Feedback
            </Button>
            <Button 
              onClick={submitFeedback} 
              disabled={submitting}
              className="flex items-center gap-2"
            >
              <MessageSquare className="w-4 h-4" />
              {submitting ? 'Submitting...' : 'Submit Feedback'}
            </Button>
          </div>
        </div>
      )}
    </div>
  );
}