// ActivityRecorder Popup Script
console.log('ActivityRecorder popup loaded');

// DOM elements
const elements = {
  statusIndicator: document.getElementById('status-indicator'),
  statusText: document.getElementById('status-text'),
  recordingStats: document.getElementById('recording-stats'),
  duration: document.getElementById('duration'),
  activityCount: document.getElementById('activity-count'),
  
  idleControls: document.getElementById('idle-controls'),
  recordingControls: document.getElementById('recording-controls'),
  taskDescriptionControls: document.getElementById('task-description-controls'),
  processingState: document.getElementById('processing-state'),
  planDisplayState: document.getElementById('plan-display-state'),
  
  startRecording: document.getElementById('start-recording'),
  stopRecording: document.getElementById('stop-recording'),
  generatePlan: document.getElementById('generate-plan'),
  discardRecording: document.getElementById('discard-recording'),
  newRecording: document.getElementById('new-recording'),
  viewRecipes: document.getElementById('view-recipes'),
  viewFullRecipe: document.getElementById('view-full-recipe'),
  
  taskDescription: document.getElementById('task-description'),
  planSummary: document.getElementById('plan-summary'),
  stepCount: document.getElementById('step-count'),
  confidenceScore: document.getElementById('confidence-score'),
  
  portalLink: document.getElementById('portal-link'),
  helpLink: document.getElementById('help-link'),
  settingsLink: document.getElementById('settings-link')
};

// State management
let currentState = 'idle';
let recordingData = null;
let durationTimer = null;

// Initialize popup
async function initialize() {
  console.log('Initializing popup...');
  
  // Clear any existing timer to prevent duplication
  if (durationTimer) {
    clearInterval(durationTimer);
    durationTimer = null;
  }
  
  // Get current recording status
  try {
    const response = await sendMessage({ type: 'GET_STATUS' });
    if (response.isRecording) {
      setState('recording');
      updateRecordingStats(response.activityCount);
      startDurationTimer();
    } else {
      setState('idle');
    }
  } catch (error) {
    console.error('Failed to get recording status:', error);
    setState('idle');
  }
  
  setupEventListeners();
}

// Event listeners
function setupEventListeners() {
  elements.startRecording.addEventListener('click', handleStartRecording);
  elements.stopRecording.addEventListener('click', handleStopRecording);
  elements.generatePlan.addEventListener('click', handleGeneratePlan);
  elements.discardRecording.addEventListener('click', handleDiscardRecording);
  elements.newRecording.addEventListener('click', handleNewRecording);
  elements.viewRecipes.addEventListener('click', handleViewRecipes);
  elements.viewFullRecipe.addEventListener('click', handleViewFullRecipe);
  
  elements.portalLink.addEventListener('click', handlePortalLink);
  elements.helpLink.addEventListener('click', handleHelpLink);
  elements.settingsLink.addEventListener('click', handleSettingsLink);
  
  // Auto-resize textarea
  elements.taskDescription.addEventListener('input', function() {
    this.style.height = 'auto';
    this.style.height = this.scrollHeight + 'px';
  });
}

// Action handlers
async function handleStartRecording() {
  console.log('▶️ Starting recording...');
  
  try {
    const response = await sendMessage({ type: 'START_RECORDING' });
    if (response.success) {
      setState('recording');
      startDurationTimer();
      updateRecordingStats(0);
    } else {
      throw new Error('Failed to start recording');
    }
  } catch (error) {
    console.error('Start recording failed:', error);
    showError('Failed to start recording: ' + error.message);
  }
}

async function handleStopRecording() {
  console.log('⏹️ Stopping recording...');
  
  try {
    const response = await sendMessage({ type: 'STOP_RECORDING' });
    if (response.success) {
      recordingData = response.sessionData;
      setState('taskDescription');
      stopDurationTimer();
    } else {
      throw new Error('Failed to stop recording');
    }
  } catch (error) {
    console.error('Stop recording failed:', error);
    showError('Failed to stop recording: ' + error.message);
  }
}

async function handleGeneratePlan() {
  const description = elements.taskDescription.value.trim();
  
  if (!description) {
    showError('Please describe what you accomplished during the recording.');
    return;
  }
  
  console.log('🤖 Generating plan...');
  setState('processing');
  
  try {
    const response = await sendMessage({ 
      type: 'GENERATE_PLAN',
      taskDescription: description
    });
    
    if (response.success) {
      displayPlan(response.plan);
      setState('planDisplay');
    } else {
      throw new Error(response.error || 'Failed to generate plan');
    }
  } catch (error) {
    console.error('Plan generation failed:', error);
    showError('Failed to generate plan: ' + error.message);
    setState('taskDescription');
  }
}

function handleDiscardRecording() {
  recordingData = null;
  elements.taskDescription.value = '';
  setState('idle');
}

function handleNewRecording() {
  recordingData = null;
  elements.taskDescription.value = '';
  setState('idle');
}

function handleViewRecipes() {
  // Open recipes portal in new tab
  chrome.tabs.create({ 
    url: getPortalUrl() + '/recipes'
  });
}

function handleViewFullRecipe() {
  // Open the specific recipe in portal
  if (recordingData) {
    chrome.tabs.create({ 
      url: getPortalUrl() + '/recipes/' + recordingData.sessionId
    });
  }
}

function handlePortalLink() {
  chrome.tabs.create({ 
    url: getPortalUrl()
  });
}

function handleHelpLink() {
  chrome.tabs.create({ 
    url: 'https://activityrecorder.com/help'
  });
}

function handleSettingsLink() {
  chrome.tabs.create({ 
    url: getPortalUrl() + '/settings'
  });
}

// Utility functions
function sendMessage(message) {
  return new Promise((resolve, reject) => {
    chrome.runtime.sendMessage(message, (response) => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else {
        resolve(response);
      }
    });
  });
}

function setState(newState) {
  console.log('🔄 State change:', currentState, '->', newState);
  currentState = newState;
  
  // Hide all state-specific elements
  elements.idleControls.classList.add('hidden');
  elements.recordingControls.classList.add('hidden');
  elements.taskDescriptionControls.classList.add('hidden');
  elements.processingState.classList.add('hidden');
  elements.planDisplayState.classList.add('hidden');
  elements.recordingStats.classList.add('hidden');
  
  // Show relevant elements for current state
  switch (newState) {
    case 'idle':
      elements.idleControls.classList.remove('hidden');
      elements.statusIndicator.className = 'status-indicator idle';
      elements.statusText.textContent = 'Ready to record';
      break;
      
    case 'recording':
      elements.recordingControls.classList.remove('hidden');
      elements.recordingStats.classList.remove('hidden');
      elements.statusIndicator.className = 'status-indicator recording';
      elements.statusText.textContent = 'Recording...';
      break;
      
    case 'taskDescription':
      elements.taskDescriptionControls.classList.remove('hidden');
      elements.statusIndicator.className = 'status-indicator idle';
      elements.statusText.textContent = 'Describe your task';
      elements.taskDescription.focus();
      break;
      
    case 'processing':
      elements.processingState.classList.remove('hidden');
      elements.statusIndicator.className = 'status-indicator recording';
      elements.statusText.textContent = 'Generating guide...';
      break;
      
    case 'planDisplay':
      elements.planDisplayState.classList.remove('hidden');
      elements.statusIndicator.className = 'status-indicator idle';
      elements.statusText.textContent = 'Guide ready!';
      break;
  }
}

function startDurationTimer() {
  // Clear any existing timer first
  if (durationTimer) {
    clearInterval(durationTimer);
  }
  
  let seconds = 0;
  durationTimer = setInterval(() => {
    seconds++;
    const minutes = Math.floor(seconds / 60);
    const remainingSeconds = seconds % 60;
    elements.duration.textContent = 
      `${minutes.toString().padStart(2, '0')}:${remainingSeconds.toString().padStart(2, '0')}`;
  }, 1000);
}

function stopDurationTimer() {
  if (durationTimer) {
    clearInterval(durationTimer);
    durationTimer = null;
  }
}

function updateRecordingStats(activityCount) {
  elements.activityCount.textContent = `${activityCount} activities`;
}

function displayPlan(plan) {
  if (plan.summary) {
    elements.planSummary.textContent = plan.summary;
  }
  
  if (plan.steps) {
    elements.stepCount.textContent = `${plan.steps.length} steps`;
  }
  
  if (plan.confidence) {
    const confidence = Math.round(plan.confidence * 100);
    elements.confidenceScore.textContent = `${confidence}% confidence`;
  }
}

function showError(message) {
  // Simple error display - in production, use a proper toast/notification system
  alert(message);
}

function getPortalUrl() {
  // Use Replit hosted backend URL
  return 'https://afc8f4e5-ffe6-415c-bd1e-9c62bd8b49b1-00-3adg58ea9ee6l.spock.replit.dev';
}

// Initialize when popup loads
document.addEventListener('DOMContentLoaded', initialize);

// Cleanup on popup close
window.addEventListener('beforeunload', () => {
  if (durationTimer) {
    clearInterval(durationTimer);
  }
});