// ActivityRecorder Content Script
console.log('ActivityRecorder content script loaded on:', window.location.href);

// Track if we've already initialized to prevent multiple injections
if (window.activityRecorderInjected) {
  console.log('ActivityRecorder already injected, skipping...');
} else {
  window.activityRecorderInjected = true;

  // Activity capture state
  let isMonitoring = false;
  let activityBuffer = [];

  // Check recording status on page load
  try {
    chrome.runtime.sendMessage({ type: 'GET_STATUS' }, (response) => {
      if (chrome.runtime.lastError) {
        console.warn('Extension context unavailable:', chrome.runtime.lastError.message);
        return;
      }
      if (response && response.isRecording) {
        startMonitoring();
      }
    });
  } catch (error) {
    console.warn('Failed to check recording status:', error);
  }

  // Listen for recording state changes
  try {
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      if (message.type === 'START_MONITORING') {
        startMonitoring();
        sendResponse({ success: true });
      } else if (message.type === 'STOP_MONITORING') {
        stopMonitoring();
        sendResponse({ success: true });
      }
    });
  } catch (error) {
    console.warn('Failed to set up message listener:', error);
  }

  function startMonitoring() {
    if (isMonitoring) return;
    
    console.log('Starting activity monitoring on:', window.location.href);
    isMonitoring = true;
    
    // Capture initial page load activity
    captureActivity({
      action: 'page_load',
      url: window.location.href,
      title: document.title,
      timestamp: Date.now()
    });

    // Add event listeners
    document.addEventListener('click', handleClick, true);
    document.addEventListener('submit', handleFormSubmit, true);
    document.addEventListener('input', handleInput, true);
    document.addEventListener('scroll', handleScroll, true);
    window.addEventListener('beforeunload', handlePageUnload, true);
  }

  function stopMonitoring() {
    if (!isMonitoring) return;
    
    console.log('⏹️ Stopping activity monitoring');
    isMonitoring = false;
    
    // Remove event listeners
    document.removeEventListener('click', handleClick, true);
    document.removeEventListener('submit', handleFormSubmit, true);
    document.removeEventListener('input', handleInput, true);
    document.removeEventListener('scroll', handleScroll, true);
    window.removeEventListener('beforeunload', handlePageUnload, true);
  }

  // Event handlers
  function handleClick(event) {
    if (!isMonitoring) return;
    
    const element = event.target;
    const activity = {
      action: 'click',
      url: window.location.href,
      title: document.title,
      element: {
        tagName: element.tagName,
        id: element.id || null,
        className: element.className || null,
        textContent: element.textContent?.substring(0, 100) || null,
        href: element.href || null
      },
      coordinates: {
        x: event.clientX,
        y: event.clientY
      }
    };
    
    captureActivity(activity);
  }

  function handleFormSubmit(event) {
    if (!isMonitoring) return;
    
    const form = event.target;
    const formData = {};
    
    // Safely capture form data (excluding sensitive fields)
    try {
      const formDataObj = new FormData(form);
      for (let [key, value] of formDataObj.entries()) {
        // Skip sensitive fields
        if (!key.toLowerCase().includes('password') && 
            !key.toLowerCase().includes('ssn') &&
            !key.toLowerCase().includes('card')) {
          formData[key] = typeof value === 'string' ? value.substring(0, 100) : '[File]';
        }
      }
    } catch (error) {
      console.warn('Error capturing form data:', error);
    }
    
    const activity = {
      action: 'form_submit',
      url: window.location.href,
      title: document.title,
      element: {
        tagName: form.tagName,
        id: form.id || null,
        className: form.className || null,
        action: form.action || null,
        method: form.method || 'GET'
      },
      formData: formData
    };
    
    captureActivity(activity);
  }

  function handleInput(event) {
    if (!isMonitoring) return;
    
    // Throttle input events to avoid spam
    if (event.target.dataset.lastInputCapture && 
        Date.now() - parseInt(event.target.dataset.lastInputCapture) < 2000) {
      return;
    }
    event.target.dataset.lastInputCapture = Date.now().toString();
    
    const element = event.target;
    
    // Skip sensitive inputs
    if (element.type === 'password' || 
        element.type === 'hidden' ||
        element.name?.toLowerCase().includes('password')) {
      return;
    }
    
    const activity = {
      action: 'input',
      url: window.location.href,
      title: document.title,
      element: {
        tagName: element.tagName,
        type: element.type || null,
        id: element.id || null,
        name: element.name || null,
        placeholder: element.placeholder || null,
        valueLength: element.value?.length || 0
      }
    };
    
    captureActivity(activity);
  }

  function handleScroll(event) {
    if (!isMonitoring) return;
    
    // Throttle scroll events
    if (window.lastScrollCapture && Date.now() - window.lastScrollCapture < 3000) {
      return;
    }
    window.lastScrollCapture = Date.now();
    
    const activity = {
      action: 'scroll',
      url: window.location.href,
      title: document.title,
      scrollPosition: {
        x: window.scrollX,
        y: window.scrollY
      },
      documentHeight: document.documentElement.scrollHeight,
      viewportHeight: window.innerHeight
    };
    
    captureActivity(activity);
  }

  function handlePageUnload(event) {
    if (!isMonitoring) return;
    
    const activity = {
      action: 'page_unload',
      url: window.location.href,
      title: document.title,
      timeOnPage: Date.now() - (window.pageLoadTime || Date.now())
    };
    
    captureActivity(activity);
  }

  // Activity capture and transmission
  function captureActivity(activity) {
    if (!isMonitoring) return;
    
    // Add timestamp and session info
    activity.timestamp = Date.now();
    activity.userAgent = navigator.userAgent;
    activity.viewport = {
      width: window.innerWidth,
      height: window.innerHeight
    };
    
    // Send to background script for processing
    chrome.runtime.sendMessage({
      type: 'ACTIVITY_CAPTURED',
      activity: activity
    }, (response) => {
      if (chrome.runtime.lastError) {
        console.warn('⚠️ Failed to send activity, queuing:', chrome.runtime.lastError);
        // Extension context invalidated - reload page to reinitialize
        if (chrome.runtime.lastError.message?.includes('Extension context invalidated')) {
          console.warn('🔄 Extension context invalidated - please refresh the page to continue recording');
        }
      } else {
        console.log('✅ Activity sent successfully to background script');
      }
    });
    
    console.log('📊 Activity captured:', activity.action, 'on', activity.url?.substring(0, 50) + '...');
  }

  // Page load tracking
  window.pageLoadTime = Date.now();
  
  // Monitor for dynamic content changes (SPA navigation)
  let lastUrl = window.location.href;
  const observer = new MutationObserver(() => {
    if (lastUrl !== window.location.href) {
      lastUrl = window.location.href;
      if (isMonitoring) {
        captureActivity({
          action: 'spa_navigation',
          url: window.location.href,
          title: document.title
        });
      }
    }
  });
  
  observer.observe(document.body, {
    childList: true,
    subtree: true
  });

  console.log('✅ ActivityRecorder content script initialized');
}